// LinksManager.cpp: implementation of the CLinksManager class.
//
//////////////////////////////////////////////////////////////////////

#include "pch.h"
#include "Adrenalin.h"
#include "Temp.hpp"
#include "ConfigWriter.h"

#include "libIncludes\MiniSAX.h"
#include "LinksSAX.h"

using	miniSAX::CDataSource;
using	miniSAX::CMiniSAXParser;
using	miniSAX::CMiniSAXException;

#include "LinksManager.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


LPCTSTR	LINKS_CFG_NAME = TEXT("links.xml");


CLinksManager::CLinksManager()
	: linksChanged(false), loaded(false)
{
}


void CLinksManager::load() throw(CBadConfigException)
{
	// we don't want to load more than once
	loaded = true;

    TCHAR	linksCfgFullName[MAX_PATH];

    lstrcpy( linksCfgFullName, CAdrenalin::getStartupPath() );
    lstrcat( linksCfgFullName, LINKS_CFG_NAME );

	try {
		CMiniSAXParser			parser;
		CLinksContentHandler	linksContentHandler( &parser );
		CDataSource				dataSource( linksCfgFullName );

		parser.setContentHandler( &linksContentHandler );

		parser.parse( &dataSource );

	} catch (miniSAX::CIOException& ioEx) {
		throw CBadConfigException( tstring("Failed to read 'links.xml': ") + ioEx.getMessage() );
	} catch (CMiniSAXException& saxEx) {
		// prepare exception message
		tstring	msg("Failed to read 'links.xml':");
		if (saxEx.getColumn() >= 0 && saxEx.getRow() >= 0) {
			char	buffer[256];
			sprintf( buffer, " [col:%d,row:%d] ", saxEx.getColumn(), saxEx.getRow() );
			msg += buffer;
		} else {
			msg += " [no position info] ";
		}
		msg += saxEx.getMessage();

		throw CBadConfigException( msg );
	}
}


CLinksManager::~CLinksManager()
{
}


PLink CLinksManager::getLink(const SFTNAddress& addr) {
	vector<PLink>::const_iterator	it;
	for (it = links.begin(); it != links.end(); it++) {
		SFTNAddress	linkAddr = (*it)->getAddress();
		if (linkAddr == addr) {
			return (*it);
		}
	}

	return NULL;
}


/**************************
  CLinksManager::CIterator
 **************************/


CLinksManager::CIterator::CIterator() {
	manager = NULL;
	index   = -1;
}


CLinksManager::CIterator::CIterator(CLinksManager* aManager, int anIndex) {
	manager = aManager;
	index   = anIndex;
}


CLinksManager::CIterator::CIterator(const CLinksManager::CIterator& instance) {
	manager = instance.manager;
	index   = instance.index;
}


CLinksManager::CIterator& CLinksManager::CIterator::operator =(const CLinksManager::CIterator& instance) {
	manager = instance.manager;
	index   = instance.index;

	return (*this);
}


bool CLinksManager::CIterator::operator ==(const CLinksManager::CIterator& instance) {
	return (manager == instance.manager && index == instance.index);
}


bool CLinksManager::CIterator::operator !=(const CLinksManager::CIterator& instance) {
	return (manager != instance.manager || index != instance.index);
}


CLinksManager::CIterator& CLinksManager::CIterator::operator ++() {
	++index;

	return (*this);
}


PLink CLinksManager::CIterator::operator *() {
	return manager->getLink( index );
}


CLinksManager::CIterator CLinksManager::begin() {
	CLinksManager::CIterator	it( this, 0 );

	return it;
}


CLinksManager::CIterator CLinksManager::end() {
	CLinksManager::CIterator	it( this, links.size() );

	return it;
}


void CLinksManager::saveLinks() throw(CIOException,CRuntimeException) {
    TCHAR	cfgName[MAX_PATH];

    lstrcpy( cfgName, CAdrenalin::getStartupPath() );
    lstrcat( cfgName, LINKS_CFG_NAME );	

	CConfigWriter	out( cfgName );

	CConfigWriter::CEntityResolver	er;
	CConfigWriter::COemManipulator	oem;

	try {

		out << "<links>\n";
		CIterator	it = begin();
		while (it != end()) {
			PLink	link = (*it);

			out << "    <link>\n";
			// name is optional
			if (link->getName().size() > 0) {
				out << "        <name>" << er << oem << link->getName() << "</name>\n";
			}

			// address
			out << "        <address>" << er << link->getAddress().toShortestString( g_vars.m_addrMain ) << "</address>\n";

			// use AKA
			if (link->isAKASpecified()) {
				out << "        <useAKA>" << er << link->getOurAddress().toString() << "</useAKA>\n";
			}

			// outbound
			out << "        <outbound type=\"";
			switch (link->getOutboundType()) {
				case OT_BINKLEY:
					out << "binkley";
					break;
				case OT_FILE_BOXES:
					out << "fileBoxes";
					break;
			}
			out << "\" flavour=\"";
			switch (link->getFlavourType()) {
			case FT_NORMAL:
				out << "normal";
				break;
			case FT_CRASH:
				out << "crash";
				break;
			case FT_DIRECT:
				out << "direct";
				break;
			case FT_HOLD:
				out << "hold";
				break;
			case FT_IMMEDIATE:
				out << "immediate";
				break;
			default:
				throw CRuntimeException( "Unknown flavour type" );
			}
			out << "\" />\n";

			// security
			out << "        <security>\n";
			out << "            <password>" << er << link->getPassword() << "</password>\n";
			if (link->getSendPriority() > 0) {
				out << "            <sendPriority>" << link->getSendPriority() << "</sendPriority>\n";
			}
			if (link->getReceivePriority() > 0) {
				out << "            <receivePriority>" << link->getReceivePriority() << "</receivePriority>\n";
			}
			// print groups
			int	groupsCount = link->getGroupsCount();
			for (int i = 0; i < groupsCount; i++) {
				out << "            <group>" << link->getGroup(i) << "</group>\n";
			}
			out << "        </security>\n";

			// area creation
			if (link->canCreateAreas()) {
				out << "        <allowAreaCreation>\n";
				out << "            <defaultGroup>" << er << oem << link->getDefaultGroup() << "</defaultGroup>\n";
				if (link->getAutoCreatedPathName().empty() == false) {
					out << "            <path>" << er << link->getAutoCreatedPathName() << "</path>\n";
				}
				if (link->getAutocreatedSendPriority() > 0) {
					out << "            <sendPriority>" << link->getAutocreatedSendPriority() << "</sendPriority>\n";
				}
				if (link->getAutocreatedReceivePriority() > 0) {
					out << "            <receivePriority>" << link->getAutocreatedReceivePriority() << "</receivePriority>\n";
				}
				out << "        </allowAreaCreation>\n";
			}

			// status
			if (link->getActivity() != ACTIVITY_USUAL) {
				out << "        <status>";
				switch (link->getActivity()) {
				case ACTIVITY_PASSIVE:
					out << "passive";
					break;
				case ACTIVITY_UNAVAILABLE:
					out << "unavailable";
					break;
				default:
					throw CRuntimeException( "Unknown activity type" );
				}
				out << "</status>\n";
			}

			// robot name
			if (link->getRobotName().length() > 0) {
				out << "        <robotName>" << er << oem << link->getRobotName() << "</robotName>\n";
			}

			// available areas
			if (link->getAvailAreasFileName().length() > 0) {
				out << "        <availAreas>" << er << link->getAvailAreasFileName() << "</availAreas>\n";
			}

			out << "    </link>\n";

			++it;
		}
		out << "</links>\n";

	} catch (...) {
		out.cancel();
		throw;
	}

	out.close();
}


bool CLinksManager::parseLinkAddress(const char* s, SFTNAddress& address) {
    TCHAR	szFullAddr[24+MAX_DOMAIN_SIZE];
    TCHAR	szTempStr[24+MAX_DOMAIN_SIZE];
    TCHAR	szDomain[MAX_DOMAIN_SIZE];
    bool	fColonPresent, fSlashPresent, fPointPresent, fDogPresent;
    int		DogPos;
    int		i;

    if (lstrlen(s) >= 24+MAX_DOMAIN_SIZE)
        return (false);
    
    fColonPresent = fSlashPresent = fPointPresent = fDogPresent = false;
    i = 0;
    while (s[i] != _T('\0')) {
        switch (s[i]) {
        case _T(':'):
            fColonPresent = true;
            break;
        case _T('/'):
            fSlashPresent = true;
            break;
        case _T('.'):
            fPointPresent = true;
            break;
        case _T('@'):
            if (!fDogPresent) {
                fDogPresent = true;
                DogPos      = i;
            }
            break;
        }
        i++;
    }
    lstrcpy( szTempStr, s );
    if (fDogPresent && !fPointPresent) {
        i = lstrlen( szTempStr + DogPos + 1);
        if (i > MAX_DOMAIN_SIZE-1)
            i = MAX_DOMAIN_SIZE-1;
        lstrcpyn( szDomain, szTempStr + DogPos + 1, i+1 );
        szTempStr[DogPos] = _T('\0');
    }
    if (!fSlashPresent) {
        _stprintf( szFullAddr, _T("%lu/%s"), g_vars.m_addrMain.Net, szTempStr );
        lstrcpy( szTempStr, szFullAddr );
    }
    if (!fColonPresent) {
        _stprintf( szFullAddr, _T("%lu:%s"), g_vars.m_addrMain.Zone, szTempStr );
    }
    else
        lstrcpy( szFullAddr, szTempStr );

    if (!ParseFTNAddress( szFullAddr, address ))
        return (false);
    
    if (!fDogPresent) {
        lstrcpy( address.Domain, g_vars.m_addrMain.Domain );
    }
    else if (!fPointPresent) {
        lstrcpy( address.Domain, szDomain );
    }

    return (*(address.Domain) != _T('\0'));
}


void CLinksManager::chooseAKA(PLink link) {
    int		i;
    int		nChoosed;
    int		nMaxSameness, nSameness;
    // sameness coefficient is:
    // if zones are same == 1
    // if nets too       == 2
    // if nodes too      == 3
    // if points too     == 4

    nChoosed = MAIN_ADDRESS;
    nMaxSameness = 0;
    if (lstrcmpi( g_vars.m_addrMain.Domain, link->getAddress().Domain ) == 0) {
        if (g_vars.m_addrMain.Zone == link->getAddress().Zone) {
            nMaxSameness++;
            if (g_vars.m_addrMain.Net == link->getAddress().Net) {
                nMaxSameness++;
                if (g_vars.m_addrMain.Node == link->getAddress().Node) {
                   nMaxSameness++;
                   if (g_vars.m_addrMain.Point == link->getAddress().Point)
                       nMaxSameness++;
                }
            }
        }
    }
    for (i = 0; i < g_vars.m_vAKAAddr.size(); i++) {
        if (lstrcmpi( g_vars.m_vAKAAddr[i].Domain, link->getAddress().Domain ) == 0)
        {
            nSameness = 0;
            if (g_vars.m_vAKAAddr[i].Zone == link->getAddress().Zone) {
                nSameness++;
                if (g_vars.m_vAKAAddr[i].Net == link->getAddress().Net) {
                    nSameness++;
                    if (g_vars.m_vAKAAddr[i].Node == link->getAddress().Node) {
                       nSameness++;
                       if (g_vars.m_vAKAAddr[i].Point == link->getAddress().Point)
                           nSameness++;
                    }
                }
            }
            if (nSameness > nMaxSameness) {
                nChoosed     = i;
                nMaxSameness = nSameness;
            }
        }
    }
    // fill OurAddr field
	if (nChoosed == MAIN_ADDRESS) {
		link->setOurAddress( g_vars.m_addrMain );
	}
	else {
		link->setOurAddress( g_vars.m_vAKAAddr[nChoosed] );
	}
}


unsigned long CLinksManager::getGroupNumber(const tstring& groupName) {	
	unsigned long	number = 1;
	for (char letter = 'A';	letter <= 'Z'; letter++) {
		if (letter == groupName[0]) {
			return number;
		}
		number <<= 1;
	}

	return 0;
}


const tstring CLinksManager::getGroupName(unsigned long groupNumber) {
	unsigned long	number = 1;
	for (char letter = 'A';	letter <= 'Z'; letter++) {
		if (number == groupNumber) {
			char	groupName[2];
			groupName[0] = letter;
			groupName[1] = '\0';
			return groupName;
		}
		number <<= 1;
	}

	return "";
}
